//
// libchmxx - c++ interface for chmlib
// Copyright (C) 2003, 2008 by Indrek Mandre <indrek@mare.ee>
// http://www.mare.ee/indrek/libchmxx/
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iostream>
#include <cstdlib>
#include <iterator>
#include <algorithm>
#include <list>

#include <fstream>

#include <sys/types.h>
#include <sys/stat.h>

#include <cerrno>
#include <cstring>

#include "chmxx.h"

using namespace std;
using namespace chm;

namespace {

void print_topics_tree (const chm_topics_tree *tree, int level = 0)
{
  if ( level != 0 )
      cout << string(level * 4, ' ') << tree->title << endl;

  for_each (tree->children.begin(), tree->children.end(),
      bind2nd(ptr_fun(print_topics_tree), level + 1));
}

void error (const string& pname, const string& err = "") {
  if ( err == "" ) {
      cerr << pname << ": Usage: " << pname << " <file.chm> list|get|listall|export|stats|topics|index|indexall [<path|text>]" << endl;
  } else {
      cerr << pname << ": " << err << endl;
  }
  exit(1);
}

void do_export (const string& pname, const chmfile& chm, const string& root, const string& p)
{
  if ( mkdir(root.c_str(), 0755) == -1 && errno != EEXIST)
      error (pname, string() + "Unable to create the target directory [" + root + "]: " + strerror(errno));

  list<string> dirs;
  chm.readdir(p, dirs, chmfile::dirs);

  list<string> files;
  chm.readdir(p, files, chmfile::files);

  for ( list<string>::iterator it = dirs.begin(); it != dirs.end(); it++ ) {
      do_export (pname, chm, root + *it, p + *it);
  }

  for ( list<string>::iterator it = files.begin(); it != files.end(); it++ ) {
      string path = root + *it;
      ofstream out(path.c_str());
      if ( !out )
          error (pname, string() + "Unable to open export file for writing: " + strerror (errno));
      chm.read (p + *it, out);
      out.close();
  }
}

void list_all (const string& pname, const chmfile& chm, const string& root = "/")
{
  list<string> l;
  chm.readdir (root, l);
  for (list<string>::iterator it = l.begin(); it != l.end(); it++ ) {
      string& n = *it;
      cout << root << n << endl;
      if ( n[n.size()-1] == '/' ) list_all (pname, chm, root + n);
  }
}

}

int main(int argc, char *argv[])
{
  if ( argc < 3 ) error(argv[0]);

  chmfile chm(argv[1]);
  if ( !chm ) error (argv[0], "Invalid chm file");

  string cmd = argv[2];

  if (  cmd == "list" ) {
      if ( argc != 4 ) error(argv[0]);
      list<string> dc;
      chm.readdir (argv[3], dc);
      ostream_iterator<string> out(cout, "\n");
      copy(dc.begin(), dc.end(), out);
      return 0;
  } else if ( cmd == "get" ) {
      if ( argc != 4 ) error(argv[0]);
      streambuf *sb = chm.open(argv[3]);
      if ( !sb )
          error (argv[0], "Could not find given path inside the archive");
      cout << sb;
      delete sb;
      return 0;
  } else if ( cmd == "listall" ) {
      list_all (argv[0], chm);
      return 0;
  } else if ( cmd == "stats" ) {
      cout << "---Statistics---" << endl;
      cout << "     Title: " << chm.get_title() << endl;
      cout << " Generator: " << chm.get_generator() << endl;
      cout << "      Home: " << chm.get_home_file() << endl;
      cout << "    Topics: " << chm.get_topics_file() << endl;
      cout << "     Index: " << chm.get_index_file() << endl;
      return 0;
  } else if ( cmd == "topics" ) {
      if ( chm.get_topics_tree() ) {
          cout << "Topics tree:" << endl;
          print_topics_tree (chm.get_topics_tree());
      } else {
          cout << "No topics tree found" << endl;
      }
      return 0;
  } else if ( cmd == "export" ) {
      if ( argc != 4 ) error(argv[0]);
      cout << "Exporting.." << endl;
      string w = argv[3];
      if ( !w.empty() && w[w.size() - 1] != '/' ) w += '/';
      do_export (argv[0], chm, w, "/");
      return 0;
  } else if ( cmd == "index" ) {
      if ( argc != 4 ) error(argv[0]);

      list<chm_search_match> f;
      chm.cache_search_database();
      chm.search_index(argv[3], f);

      for ( list<chm_search_match>::iterator it = f.begin(); it != f.end(); it++ )
          for ( size_t i = 0; i < it->documents.size(); i++ )
              cout << it->key << "|" << it->is_title << "|" << it->documents[i].offsets.size() << "|"
                  << it->documents[i].path << "|" << it->documents[i].title << endl;

      return 0;
  } else if ( cmd == "indexall" ) {
      if ( argc != 4 ) error(argv[0]);

      list<chm_search_match> f;
      chm.cache_search_database();
      if ( !chm.search_index(argv[3], f, false, false) )
          error (argv[0], "search failed!");

      for ( list<chm_search_match>::iterator it = f.begin(); it != f.end(); it++ )
          for ( size_t i = 0; i < it->documents.size(); i++ )
              cout << it->key << "|" << it->is_title << "|" << it->documents[i].offsets.size() << "|"
                  << it->documents[i].path << "|" << it->documents[i].title << endl;

      return 0;
  }

  error (argv[0], "Invalid second argument");
}
