// 
// XML4SableCC - XML generator for the SableCC
// Copyright (C) 2001-2003 Indrek Mandre <indrek@mare.ee>
// All rights reserved.
// 
// See the file "AUTHORS" for the name of all copyright holders.
// 
// This program is free software; you can redistribute it and/or modify it
// under the terms of the GNU Lesser General Public License as published
// by the Free Software Foundation; either version 2.1 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this program in the file "COPYING-LESSER"; if not,
// write to the Free Software Foundation, Inc., 59 Temple Place,
// Suite 330, Boston, MA  02111-1307  USA
// 
// For more information and latest updates please see
// http://www.mare.ee/indrek/sablecc/
// 

package ee.mare.indrek.sablecc.xml;

import java.util.*;
import java.io.*;
import java.text.SimpleDateFormat;

import org.sablecc.sablecc.ParserGenerator;
import org.sablecc.sablecc.ParserData;

public class XMLGenerator implements ParserGenerator {
  public void init (String arguments) throws Exception
  {
  }

  public String getName ()
  {
    return "XML generator for SableCC";
  }

  public String getInfo ()
  {
    return "XML for SableCC parser generator\n" +
        "Copyright (C) 2003 Indrek Mandre <indrek@mare.ee>";
  }

  public String getHelp ()
  {
    return "";
  }

  public void generate (ParserData data, String path) throws Exception
  {
    BufferedWriter out = new BufferedWriter (new FileWriter (new File(path, "parser.xml")));

    out.write ("<?xml version=\"1.0\"?>\n");
    out.write ("<parser package=\"" + escape (data.package_name) + "\" generator=\"SableCC3\" date=\"" +
        new SimpleDateFormat( "yyyy-MM-dd'T'HH:mm:ss.S zzz" ).format(new Date()) + "\">\n");

    out.write ("  <tokens>\n");
    for ( Iterator it = data.token_list.iterator(); it.hasNext(); ) {
        ParserData.Token token = (ParserData.Token)it.next();
        out.write ("    <token name=\"" + escape(token.name) + "\" ename=\"" + escape(token.ename) + "\"");
        if ( token.value != null ) out.write (" value=\"" + escape(token.value) + "\"");
        if ( token.index != -1 ) out.write (" parser_index=\"" + token.index + "\"");

        if ( token.transitions.size() == 0 ) {
            out.write ("/>\n");
        } else {
            out.write (">\n");
            for ( Iterator jt = token.transitions.entrySet().iterator(); jt.hasNext(); ) {
                Map.Entry entry = (Map.Entry)jt.next();
                out.write ("      <transition from=\"" + (String)entry.getKey() + "\" to=\"" + (String)entry.getValue() + "\"/>\n");
            }
            out.write ("    </token>\n");
        }
    }
    out.write ("    <eof parser_index=\"" + data.eof_index + "\"/>\n");
    out.write ("  </tokens>\n");

    out.write ("  <lexer_data>\n");
    for ( Iterator it = data.state_list.iterator(); it.hasNext(); ) {
        ParserData.LexerState state = (ParserData.LexerState)it.next();
        out.write ("    <state name=\"" + escape(state.name) + "\" id=\"" + state.id + "\"/>\n");
    }

    out.write ("    <accept_table>\n");
    for ( Enumeration it = data.lexer_accept.elements(); it.hasMoreElements(); ) {
        Vector row = (Vector)it.nextElement();
        out.write ("      <state>\n");
        out.write ("        ");
        for ( Enumeration jt = row.elements(); jt.hasMoreElements(); ) {
            out.write ("<i>" + ((Number)jt.nextElement()).intValue() + "</i>");
        }
        out.write ("\n");
        out.write ("      </state>\n");
    }
    out.write ("    </accept_table>\n");

    out.write ("    <goto_table>\n");
    for ( Enumeration it = data.lexer_goto_table.elements(); it.hasMoreElements(); ) {
        Vector state = (Vector)it.nextElement();
        out.write ("      <state>\n");
        for ( Enumeration jt = state.elements(); jt.hasMoreElements(); ) {
            Vector row = (Vector)jt.nextElement();
            out.write ("        <row>\n");
            for ( Enumeration kt = row.elements(); kt.hasMoreElements(); ) {
                int[] elem = (int[])kt.nextElement();
                out.write ("          <goto low=\"" + elem[0] + "\" high=\"" + elem[1] + "\" state=\"" + elem[2] + "\"/>\n");
            }
            out.write ("        </row>\n");
        }
        out.write ("      </state>\n");
    }
    out.write ("    </goto_table>\n");

    out.write ("  </lexer_data>\n");

    out.write ("  <productions>\n");

    for ( Iterator it = data.production_list.iterator(); it.hasNext(); ) {
        ParserData.Production prod = (ParserData.Production)it.next();
        out.write ("    <production name=\"" + escape(prod.name) + "\" ename=\"" + escape(prod.ename) + "\">\n");
        for ( Iterator jt = prod.alts.iterator(); jt.hasNext(); ) {
            ParserData.Alt alt = (ParserData.Alt)jt.next();
            out.write ("      <alt name=\"" + escape(alt.name) + "\" ename=\"" + escape(alt.ename) + "\">\n");
            for ( Iterator kt = alt.elems.iterator(); kt.hasNext(); ) {
                ParserData.Elem elem = (ParserData.Elem)kt.next();
                out.write ("        <elem name=\"" + escape(elem.name) + "\" ename=\"" + escape(elem.ename) + "\" type=\"" + escape(elem.type) + "\" etype=\"" + elem.etype + "\"");
                if ( elem.modifier != null ) out.write (" modifier=\"" + escape(elem.modifier) + "\"");
                if ( elem.is_token ) out.write (" is_token=\"true\"");
                if ( elem.is_list ) out.write (" is_list=\"true\"");
                out.write ("/>\n");
            }
            out.write ("      </alt>\n");
        }
        out.write ("    </production>\n");
    }

    out.write ("  </productions>\n");

    String cmd2str[] = new String[] {"POP", "FETCHNODE", "FETCHLIST", "ADDNODE", "ADDLIST",
      "MAKELIST", "MAKENODE", "RETURNNODE", "RETURNLIST"};

    out.write ("  <rules>\n");

    for ( Iterator it = data.rules_list.iterator(); it.hasNext(); ) {
        ParserData.Rule rule = (ParserData.Rule)it.next();
        out.write ("    <rule ename=\"" + escape(rule.ename) + "\" index=\"" + rule.index + "\" leftside=\"" + rule.leftside + "\">\n");
        for ( Iterator jt = rule.actions.iterator(); jt.hasNext(); ) {
            ParserData.Action action = (ParserData.Action)jt.next();
            out.write ("      <action cmd=\"" + cmd2str[action.cmd] + "\"");
            if ( action.result != null ) out.write (" result=\"" + escape(action.result) + "\"");
            if ( action.index != -1 ) out.write (" index=\"" + action.index + "\"");
            if ( action.etype != null ) out.write (" etype=\"" + escape(action.etype) + "\"");
            switch (action.cmd) {
                case ParserData.Action.FETCHNODE:
                case ParserData.Action.FETCHLIST:
                    out.write (" from=\"" + escape(action.arg[0]) + "\"/>\n");
                    continue;
                case ParserData.Action.RETURNNODE:
                    if ( action.arg[0] == null )
                        out.write (" null=\"true\"/>\n");
                    else
                        out.write (" node=\"" + escape(action.arg[0]) + "\"/>\n");
                    continue;
                case ParserData.Action.RETURNLIST:
                    out.write (" list=\"" + escape(action.arg[0]) + "\"/>\n");
                    continue;
                case ParserData.Action.ADDNODE:
                    out.write (" tolist=\"" + escape(action.arg[0]) + "\" node=\"" + escape(action.arg[1]) + "\"/>\n");
                    continue;
                case ParserData.Action.ADDLIST:
                    out.write (" tolist=\"" + escape(action.arg[0]) + "\" fromlist=\"" + escape(action.arg[1]) + "\"/>\n");
                    continue;
            }
            if ( action.arg.length == 0 ) {
                out.write ("/>\n");
            } else {
                out.write (">\n");
                for ( int i = 0; i < action.arg.length; i++ ) {
                    if ( action.arg[i] == null )
                        out.write ("        <arg null=\"true\"/>\n");
                    else
                        out.write ("        <arg>" + escape(action.arg[i]) + "</arg>\n");
                }
                out.write ("      </action>\n");
            }
        }
        out.write ("    </rule>\n");
    }

    out.write ("  </rules>\n");

    out.write ("  <parser_data>\n");

    out.write ("    <action_table>\n");
    for ( Enumeration it = data.parser_action_table.elements(); it.hasMoreElements(); ) {
        Vector row = (Vector)it.nextElement();
        out.write ("      <row>\n");
        for ( Enumeration jt = row.elements(); jt.hasMoreElements(); ) {
            int[] action = (int[])jt.nextElement();
            out.write ("        <action from=\"" + action[0] + "\" action=\"" + action[1] + "\" to=\"" + action[2] + "\"/>\n");
        }
        out.write ("      </row>\n");
    }
    out.write ("    </action_table>\n");

    out.write ("    <goto_table>\n");
    for ( Enumeration it = data.parser_goto_table.elements(); it.hasMoreElements(); ) {
        Vector row = (Vector)it.nextElement();
        out.write ("      <row>\n");
        for ( Enumeration jt = row.elements(); jt.hasMoreElements(); ) {
            int[] action = (int[])jt.nextElement();
            out.write ("        <goto from=\"" + action[0] + "\" to=\"" + action[1] + "\"/>\n");
        }
        out.write ("      </row>\n");
    }
    out.write ("    </goto_table>\n");


    out.write ("    <errors>\n");
    out.write ("      ");
    for ( Enumeration it = data.parser_errors.elements(); it.hasMoreElements(); ) {
        out.write ("<i>" + ((Integer)it.nextElement()).intValue() + "</i>");
    }
    out.write ("\n");
    out.write ("    </errors>\n");

    out.write ("    <error_messages>\n");
    for ( Enumeration it = data.parser_error_messages.elements(); it.hasMoreElements(); ) {
        out.write ("      <msg>" + escape((String)it.nextElement()) + "</msg>\n");
    }
    out.write ("    </error_messages>\n");

    out.write ("  </parser_data>\n");

    out.write ("</parser>\n");
    out.close();
  }

  public String escape (String str)
  {
    if ( str == null ) return "";
    StringBuffer out = new StringBuffer ();
    for ( int i = 0; i < str.length(); i++ ) {
        char n = str.charAt(i);
        if ( (n >= 'a' && n <= 'z') || (n >= 'A' && n <= 'Z') || (n >= '0' && n <= '9')) {
            out.append (n);
            continue;
        }

        switch (n) {
            case '!':
            case '?':
            case '{':
            case '}':
            case '(':
            case ')':
            case '[':
            case ']':
            case ',':
            case '.':
            case ':':
            case ';':
            case '=':
            case '-':
            case '+':
            case '*':
            case '_':
            case '/':
            case '\\':
            case '|':
            case '^':
            case '%':
            case '$':
            case ' ':
            case '#':
            case '@':
            case '\'':
            case '`':
            case '~':
                out.append (n);
                break;
            default:
                out.append ("&#" + (int)n + ";");
                break;
        }
    }
    return out.toString();
  }
}

